/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.spelt;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.util.ArrayList;
import java.util.Set;
import java.util.regex.Pattern;
import org.apache.lucene.spelt.FreqData;
import org.apache.lucene.spelt.SpellWriter;
import org.apache.lucene.spelt.TRStringDistance2;
import org.apache.lucene.spelt.WordEquiv;
import org.apache.lucene.util.Hash64;
import org.apache.lucene.util.IntList;
import org.apache.lucene.util.LongSet;
import org.apache.lucene.util.PriorityQueue;
import org.apache.lucene.util.StringUtil;

public class SpellReader {
    File spellDir;
    private IntList edMapKeys;
    private IntList edMapPosns;
    private RandomAccessFile edMapFile;
    private CharsetDecoder edMapDecoder;
    private FreqData pairFreqs;
    private FreqData wordFreqs;
    private int[] freqSamples;
    private PrintWriter debugWriter = null;
    private final Pattern splitPat = Pattern.compile("\\||\n");
    private Set stopSet;
    private WordEquiv wordEquiv;

    private SpellReader() {
    }

    public static boolean isValidDictionary(File spellDir) {
        if (!spellDir.isDirectory() || !spellDir.canRead()) {
            return false;
        }
        File file = new File(spellDir, "pairs.dat");
        return file.canRead();
    }

    public static SpellReader open(File spellIndexDir) throws IOException {
        SpellReader reader = new SpellReader();
        reader.spellDir = spellIndexDir;
        reader.stopSet = null;
        reader.wordEquiv = WordEquiv.DEFAULT;
        reader.openEdmap();
        reader.loadFreqSamples();
        reader.loadWordFreqs();
        return reader;
    }

    public void setStopwords(Set set) {
        this.stopSet = set;
    }

    public void setWordEquiv(WordEquiv eq) {
        this.wordEquiv = eq;
    }

    private void openEdmap() throws IOException {
        long startTime = System.currentTimeMillis();
        File file = new File(this.spellDir, "edmap.dat");
        try {
            FileInputStream in = new FileInputStream(file);
            in.skip(file.length() - 20L);
            BufferedReader reader = new BufferedReader(new InputStreamReader(in));
            String line = reader.readLine();
            int indexPos = Integer.parseInt(line.trim());
            reader.close();
            in = new FileInputStream(file);
            in.skip(indexPos);
            reader = new BufferedReader(new InputStreamReader(in));
            line = reader.readLine();
            if (!line.equals("edMap index")) {
                throw new IOException("edmap file corrupt");
            }
            line = reader.readLine();
            int nKeys = Integer.parseInt(line);
            this.edMapKeys = new IntList(nKeys);
            this.edMapPosns = new IntList(nKeys + 1);
            int prevKey = 0;
            int pos = 0;
            for (int i = 0; i < nKeys; ++i) {
                line = reader.readLine();
                String[] tokens = this.splitPat.split(line);
                if (tokens.length != 2) {
                    throw new IOException("edmap file corrupt");
                }
                if (tokens[0].length() != 4) {
                    throw new IOException("edmap file corrupt");
                }
                int key = this.comboKey(tokens[0], 0, 1, 2, 3);
                assert (key >= prevKey) : "edmap file out of order or corrupt";
                prevKey = key;
                this.edMapKeys.add(key);
                int size = Integer.parseInt(tokens[1]);
                this.edMapPosns.add(pos);
                pos += size;
            }
            reader.close();
            if (this.edMapKeys.size() != nKeys) {
                throw new IOException("edmap file index truncated");
            }
            this.edMapPosns.add(indexPos);
        }
        catch (NumberFormatException e) {
            throw new IOException("edmap file corrupt");
        }
        this.edMapDecoder = Charset.forName("UTF-8").newDecoder();
        this.edMapFile = new RandomAccessFile(file, "r");
        if (this.debugWriter != null) {
            this.debugWriter.println("EdMap index load time: " + (System.currentTimeMillis() - startTime));
            this.debugWriter.println("  nKeys: " + this.edMapKeys.size());
        }
    }

    public void close() throws IOException {
        if (this.edMapFile != null) {
            this.edMapFile.close();
            this.edMapFile = null;
        }
    }

    public void setDebugWriter(PrintWriter w) {
        this.debugWriter = w;
    }

    private boolean readEdKey(Word orig, int key, int minFreq, LongSet checked, WordQueue queue) throws IOException {
        int idxNum = this.edMapKeys.binarySearch(key);
        if (idxNum < 0) {
            return false;
        }
        int startPos = this.edMapPosns.get(idxNum);
        int endPos = this.edMapPosns.get(idxNum + 1);
        byte[] bytes = new byte[endPos - startPos];
        this.edMapFile.seek(startPos);
        if (this.edMapFile.read(bytes) != bytes.length) {
            throw new IOException("error reading from edMap file");
        }
        String line = this.edMapDecoder.decode(ByteBuffer.wrap(bytes)).toString().trim();
        String[] tokens = this.splitPat.split(line);
        if (tokens.length < 2) {
            throw new IOException("edmap file corrupt");
        }
        if (key != this.comboKey(tokens[0], 0, 1, 2, 3)) {
            throw new IOException("edmap index incorrect");
        }
        String prev = null;
        for (int j = 1; j < tokens.length; ++j) {
            String word = tokens[j];
            if (prev != null) {
                int overlap = word.charAt(0) - 48;
                word = prev.substring(0, overlap) + word.substring(1);
            }
            prev = word;
            long hash = Hash64.hash(word);
            if (checked.contains(hash)) continue;
            checked.add(hash);
            int freq = this.wordFreqs.get(hash);
            if (freq < minFreq || orig.wordDist(word) > 4) continue;
            Word w = new Word(orig, word, freq);
            queue.insert(w);
        }
        return true;
    }

    private void findCloseWords(Word orig, int minFreq, WordQueue queue) throws IOException {
        LongSet checked = new LongSet(100);
        this.readEdKey(orig, this.comboKey(orig.word, 0, 1, 2, 3), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 0, 1, 2, 4), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 0, 1, 2, 5), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 0, 1, 3, 4), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 0, 1, 3, 5), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 0, 1, 4, 5), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 0, 2, 3, 4), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 0, 2, 3, 5), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 0, 2, 4, 5), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 0, 3, 4, 5), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 1, 2, 3, 4), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 1, 2, 3, 5), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 1, 2, 4, 5), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 1, 3, 4, 5), minFreq, checked, queue);
        this.readEdKey(orig, this.comboKey(orig.word, 2, 3, 4, 5), minFreq, checked, queue);
    }

    private int comboKey(String word, int p0, int p1, int p2, int p3) {
        int[] ch = new int[]{word.length() > p0 ? this.comboChar(word.charAt(p0)) : 32, word.length() > p1 ? this.comboChar(word.charAt(p1)) : 32, word.length() > p2 ? this.comboChar(word.charAt(p2)) : 32, word.length() > p3 ? this.comboChar(word.charAt(p3)) : 32};
        return ch[0] << 24 | ch[1] << 16 | ch[2] << 8 | ch[3] << 0;
    }

    private int comboChar(int c) {
        if (c >= 32 && (c & 0xFFFFFF80) == 0) {
            return c;
        }
        return (c = (int)((char)(c & 0x7F | 0x20))) == 124 ? 42 : c;
    }

    public boolean inDictionary(String word) throws IOException {
        return this.wordFreqs.get(word.toLowerCase()) > 0;
    }

    public synchronized String[] suggestSimilar(String str, int numSugg) throws IOException {
        Word[] suggs = this.suggestSimilar(new Word(str), numSugg + 1, 1);
        ArrayList<String> out = new ArrayList<String>();
        for (int i = 0; i < suggs.length; ++i) {
            if (suggs[i].word.equals(str)) continue;
            out.add(suggs[i].word);
        }
        return out.toArray(new String[out.size()]);
    }

    private Word[] suggestSimilar(Word word, int numSugg, int minFreq) throws IOException {
        int i;
        int queueSize = numSugg + 10;
        WordQueue queue = new WordQueue(queueSize);
        this.findCloseWords(word, minFreq, queue);
        Word[] array = new Word[Math.min(numSugg, queue.size())];
        if (this.debugWriter != null) {
            this.debugWriter.println("    Consider: ");
        }
        for (i = queue.size() - 1; i >= 0; --i) {
            Word sugg = (Word)queue.pop();
            if (this.debugWriter != null) {
                this.debugWriter.print("      ");
                sugg.debug(this.debugWriter);
            }
            if (i >= array.length) continue;
            array[i] = sugg;
        }
        if (this.debugWriter != null) {
            this.debugWriter.println("    Final suggestion(s):");
            for (i = 0; i < array.length; ++i) {
                this.debugWriter.print("      ");
                array[i].debug(this.debugWriter);
            }
        }
        return array;
    }

    public synchronized String[] suggestKeywords(String[] terms) throws IOException {
        if (terms.length == 0) {
            return null;
        }
        this.openPairFreqs();
        Phrase in = new Phrase();
        in.words = new Word[terms.length];
        in.baseScore = -0.2f;
        for (int i = 0; i < terms.length; ++i) {
            in.words[i] = new Word(null, terms[i].toLowerCase(), this.wordFreqs.get(terms[i].toLowerCase()));
        }
        in.calcScore();
        if (this.debugWriter != null) {
            this.debugWriter.append("Original: ");
            in.calcScore(this.debugWriter);
        }
        Phrase bestPhrase = in;
        if (terms.length == 1) {
            bestPhrase = this.max(bestPhrase, this.subWord(in, 0));
            bestPhrase = this.max(bestPhrase, this.subSplit(in, 0));
        } else {
            bestPhrase = this.subPairs(in);
        }
        if (this.debugWriter != null) {
            this.debugWriter.append("  Final : ");
            bestPhrase.calcScore(this.debugWriter);
        }
        String[] out = bestPhrase.toStringArray();
        boolean anyChange = false;
        for (int i = 0; i < out.length; ++i) {
            if (out[i] == null) {
                anyChange = true;
                continue;
            }
            if (this.wordEquiv.isEquivalent(terms[i], out[i])) {
                out[i] = terms[i];
                continue;
            }
            anyChange = true;
            out[i] = StringUtil.copyCase(terms[i], out[i]);
        }
        if (this.debugWriter != null) {
            this.debugWriter.flush();
        }
        if (!anyChange) {
            return null;
        }
        return out;
    }

    private Phrase subWord(Phrase in, int pos) throws IOException {
        if (this.stopSet != null && this.stopSet.contains(in.words[pos].word)) {
            return in;
        }
        int origFreq = this.wordFreqs.get(in.words[pos].word);
        Word[] suggs = this.suggestSimilar(in.words[pos], 1, origFreq + 1);
        if (suggs.length == 0) {
            return in;
        }
        Word sugg = suggs[0];
        assert (!sugg.word.equals(in.words[pos].word));
        if (sugg == in.words[pos]) {
            return in;
        }
        if (this.wordEquiv.isEquivalent(sugg.word, in.words[pos].word)) {
            return in;
        }
        Phrase out = (Phrase)in.clone();
        out.words[0] = sugg;
        out.calcScore();
        return this.max(in, out);
    }

    private Phrase max(Phrase orig, Phrase test) throws IOException {
        if (this.debugWriter != null && test.score != orig.score) {
            this.debugWriter.append(test.score > orig.score ? "    Better: " : "    Worse : ");
            test.calcScore(this.debugWriter);
        }
        if (test.score > orig.score) {
            return test;
        }
        return orig;
    }

    private Phrase subPairs(Phrase in) throws IOException {
        Phrase bestPhrase = in;
        for (int pass = 1; pass <= 2; ++pass) {
            if (this.debugWriter != null) {
                this.debugWriter.println("  ---- Pass " + pass + " ----");
                this.debugWriter.print("  Starting with: ");
                in.calcScore(this.debugWriter);
            }
            int prev = -1;
            for (int i = 0; i < in.words.length; ++i) {
                Word w = in.words[i];
                if (w == null || this.stopSet != null && this.stopSet.contains(w.word) || w.word.indexOf(32) >= 0) continue;
                if (in.words[i].orig == in.words[i]) {
                    bestPhrase = this.max(bestPhrase, this.subSplit(in, i));
                }
                if (prev >= 0 && (in.words[i].orig == in.words[i] || in.words[prev].orig == in.words[prev])) {
                    bestPhrase = this.max(bestPhrase, this.subPair(in, prev, i));
                    bestPhrase = this.max(bestPhrase, this.subJoin(in, prev, i));
                }
                prev = i;
            }
            if (in == bestPhrase) break;
            in = bestPhrase;
        }
        return bestPhrase;
    }

    private Phrase subPair(Phrase in, int pos1, int pos2) throws IOException {
        Word[] list2;
        Word word1 = in.words[pos1];
        Word word2 = in.words[pos2];
        if (this.debugWriter != null) {
            this.debugWriter.println("  subPair(" + pos1 + ", " + pos2 + "): " + in.words[pos1].word + " " + in.words[pos2].word);
        }
        int NUM_SUG = 100;
        Word[] list1 = word1.orig == word1 ? this.suggestSimilar(word1, 100, 0) : null;
        Word[] wordArray = list2 = word2.orig == word2 ? this.suggestSimilar(word2, 100, 0) : null;
        if (list1 == null || list1.length == 0) {
            list1 = new Word[]{in.words[pos1]};
        }
        if (list2 == null || list2.length == 0) {
            list2 = new Word[]{in.words[pos2]};
        }
        float bestScore = 0.0f;
        Word bestSugg1 = null;
        Word bestSugg2 = null;
        for (int p1 = 0; p1 < list1.length; ++p1) {
            boolean change1;
            Word sugg1 = list1[p1];
            boolean bl = change1 = !this.wordEquiv.isEquivalent(in.words[pos1].word, sugg1.word);
            if (!change1) {
                sugg1 = word1;
            }
            for (int p2 = 0; p2 < list2.length; ++p2) {
                boolean change2;
                Word sugg2 = list2[p2];
                boolean bl2 = change2 = !this.wordEquiv.isEquivalent(in.words[pos2].word, sugg2.word);
                if (!change2) {
                    sugg2 = word2;
                }
                if (!change1 && !change2) continue;
                float pairScore = this.scorePair(sugg1, sugg2);
                float totalScore = pairScore + sugg1.score + sugg2.score;
                if (this.debugWriter != null) {
                    this.debugWriter.format("    Pair-replace \"%s %s\" with \"%s %s\": %.2f (%.2f + %.2f + %.2f)\n", word1, word2, sugg1, sugg2, Float.valueOf(totalScore), Float.valueOf(pairScore), Float.valueOf(sugg1.score), Float.valueOf(sugg2.score));
                }
                if (!(totalScore > bestScore)) continue;
                bestScore = totalScore;
                bestSugg1 = sugg1;
                bestSugg2 = sugg2;
            }
        }
        if (bestSugg1 == null) {
            return in;
        }
        Phrase bestPhrase = (Phrase)in.clone();
        bestPhrase.words[pos1] = bestSugg1;
        bestPhrase.words[pos2] = bestSugg2;
        bestPhrase.calcScore();
        return bestPhrase;
    }

    private Phrase subSplit(Phrase in, int pos) throws IOException {
        Phrase bestPhrase = in;
        String origStr = in.words[pos].word;
        for (int i = 2; i < origStr.length() - 1; ++i) {
            String leftStr = origStr.substring(0, i);
            String rightStr = origStr.substring(i);
            int leftFreq = this.wordFreqs.get(leftStr);
            int rightFreq = this.wordFreqs.get(rightStr);
            if (leftFreq <= 0 || rightFreq <= 0) continue;
            int pairFreq = this.pairFreqs.get(leftStr, rightStr);
            if (this.debugWriter != null) {
                this.debugWriter.format("  split-replace: '%s' with '%s' '%s': freq %d\n", origStr, leftStr, rightStr, pairFreq);
            }
            Phrase testPhrase = (Phrase)in.clone();
            testPhrase.words[pos] = new Word(in.words[pos], leftStr + " " + rightStr, pairFreq + 1);
            testPhrase.calcScore();
            bestPhrase = this.max(bestPhrase, testPhrase);
        }
        return bestPhrase;
    }

    private Phrase subJoin(Phrase in, int pos1, int pos2) throws IOException {
        Word origWord = new Word(in.words[pos1].word + " " + in.words[pos2].word);
        int origFreq = this.pairFreqs.get(in.words[pos1].word, in.words[pos2].word);
        String joinedStr = in.words[pos1].word + in.words[pos2].word;
        int joinedFreq = this.wordFreqs.get(joinedStr);
        if (joinedFreq == 0) {
            return in;
        }
        if (this.debugWriter != null) {
            this.debugWriter.format("  join-replace: \"%s %s\" with \"%s\": freq %d\n", in.words[pos1].word, in.words[pos2].word, joinedStr, joinedFreq);
        }
        if (joinedFreq <= origFreq) {
            return in;
        }
        Phrase testPhrase = (Phrase)in.clone();
        testPhrase.words[pos1] = new Word(origWord, joinedStr, joinedFreq);
        testPhrase.words[pos1].score = in.words[pos1].score + in.words[pos2].score + this.scorePair(in.words[pos1], in.words[pos2]);
        testPhrase.words[pos2] = null;
        testPhrase.calcScore();
        return testPhrase;
    }

    private float scorePair(Word sugg1, Word sugg2) throws IOException {
        int origPairFreq = this.pairFreqs.get(sugg1.orig.word, sugg2.orig.word);
        int suggPairFreq = this.pairFreqs.get(sugg1.word, sugg2.word);
        if (suggPairFreq <= origPairFreq) {
            return 0.0f;
        }
        double freqFactor = ((double)suggPairFreq + 1.0) / ((double)origPairFreq + 1.0);
        float freqBoost = (float)(Math.log(freqFactor) / Math.log(100.0)) / 2.0f;
        return freqBoost;
    }

    private void loadFreqSamples() throws IOException {
        int[] res = new int[5];
        res[4] = Integer.MAX_VALUE;
        res[3] = Integer.MAX_VALUE;
        res[2] = Integer.MAX_VALUE;
        res[1] = Integer.MAX_VALUE;
        res[0] = Integer.MAX_VALUE;
        File freqSamplesFile = new File(this.spellDir, "freqSamples.dat");
        if (!freqSamplesFile.canRead()) {
            throw new IOException("Cannot open frequency samples file '" + freqSamplesFile + "'");
        }
        BufferedReader reader = new BufferedReader(new FileReader(freqSamplesFile));
        int nSamples = 0;
        int[] samples = null;
        try {
            int nTerms = Integer.parseInt(reader.readLine());
            if (nTerms >= 500) {
                nSamples = Integer.parseInt(reader.readLine());
                samples = new int[nSamples];
                for (int i = 0; i < nSamples; ++i) {
                    samples[i] = Integer.parseInt(reader.readLine());
                }
            }
        }
        catch (NumberFormatException e) {
            throw new IOException("term frequencies file corrupt");
        }
        finally {
            reader.close();
        }
        if (samples != null) {
            res[0] = samples[(int)((double)nSamples * 0.99)];
            res[1] = samples[(int)((double)nSamples * 0.9)];
            res[2] = samples[(int)((double)nSamples * 0.5)];
            res[3] = samples[(int)((double)nSamples * 0.25)];
            res[4] = samples[0];
        }
        this.freqSamples = res;
    }

    private void loadWordFreqs() throws IOException {
        File freqFile = new File(this.spellDir, "words.dat");
        if (!freqFile.canRead()) {
            throw new IOException("Cannot open word frequency file '" + freqFile + "'");
        }
        this.wordFreqs = new FreqData();
        BufferedReader reader = new BufferedReader(new FileReader(freqFile));
        try {
            String line;
            while ((line = reader.readLine()) != null) {
                String[] toks = this.splitPat.split(line);
                String word = toks[0];
                int freq = Integer.parseInt(toks[1]);
                this.wordFreqs.add(word, freq);
            }
        }
        catch (NumberFormatException e) {
            throw new IOException("term frequencies file corrupt");
        }
        finally {
            reader.close();
        }
    }

    private void openPairFreqs() throws IOException {
        if (this.pairFreqs == null) {
            this.pairFreqs = new FreqData();
            this.pairFreqs.add(new File(this.spellDir, "pairs.dat"));
        }
    }

    protected void finalize() throws Throwable {
        this.close();
    }

    private String calcMetaphone(String word) {
        String mph = SpellWriter.calcMetaphone(word);
        if (mph == null) {
            return "";
        }
        return mph;
    }

    private class Phrase
    implements Cloneable {
        Word[] words;
        float baseScore = 0.0f;
        float score;

        private Phrase() {
        }

        public Object clone() {
            try {
                Phrase out = (Phrase)super.clone();
                out.words = new Word[this.words.length];
                out.baseScore = 0.0f;
                System.arraycopy(this.words, 0, out.words, 0, this.words.length);
                return out;
            }
            catch (CloneNotSupportedException e) {
                return null;
            }
        }

        public void calcScore() throws IOException {
            this.calcScore(null);
        }

        public void calcScore(PrintWriter debugWriter) throws IOException {
            float wordScore = 0.0f;
            float pairScore = 0.0f;
            int prev = -1;
            for (int i = 0; i < this.words.length; ++i) {
                if (this.words[i] == null) continue;
                if (SpellReader.this.stopSet != null && SpellReader.this.stopSet.contains(this.words[i].word.toLowerCase())) {
                    if (debugWriter == null) continue;
                    debugWriter.append(this.words[i].word + " ");
                    continue;
                }
                wordScore += this.words[i].score;
                if (prev >= 0 && this.words[i].word.indexOf(32) < 0) {
                    pairScore += SpellReader.this.scorePair(this.words[prev], this.words[i]);
                    if (debugWriter != null) {
                        debugWriter.format("+%.2f ", Float.valueOf(SpellReader.this.scorePair(this.words[prev], this.words[i])));
                    }
                }
                prev = i;
                if (debugWriter == null) continue;
                debugWriter.format("%s[%.2f] ", this.words[i].word, Float.valueOf(this.words[i].score));
            }
            this.score = this.baseScore + wordScore + pairScore;
            if (debugWriter != null) {
                if (this.baseScore != 0.0f) {
                    debugWriter.format("... base: %.2f ", Float.valueOf(this.baseScore));
                }
                debugWriter.format("... Total: %.2f\n", Float.valueOf(this.score));
            }
        }

        public String[] toStringArray() {
            String[] out = new String[this.words.length];
            for (int i = 0; i < this.words.length; ++i) {
                out[i] = this.words[i] == null ? null : this.words[i].word;
            }
            return out;
        }
    }

    private static final class WordQueue
    extends PriorityQueue {
        WordQueue(int size) {
            this.initialize(size);
        }

        protected final boolean lessThan(Object a, Object b) {
            Word wa = (Word)a;
            Word wb = (Word)b;
            if (wa.score > wb.score) {
                return false;
            }
            if (wa.score < wb.score) {
                return true;
            }
            if (wa.freq > wb.freq) {
                return false;
            }
            return wa.freq < wb.freq;
        }
    }

    private final class Word {
        public String word;
        public Word orig;
        public int freq;
        public String metaphone;
        private TRStringDistance2 wordDist;
        private TRStringDistance2 mphDist;
        public float score;
        public float freqBoost;

        public Word(String word) throws IOException {
            this(null, word, 0);
        }

        public Word(Word inOrig, String word, int freq) throws IOException {
            this.word = word;
            this.orig = inOrig == null ? this : inOrig;
            this.freq = freq;
            this.metaphone = SpellReader.this.calcMetaphone(word);
            this.mphDist = null;
            this.wordDist = null;
            if (this.orig != this && SpellReader.this.wordEquiv.isEquivalent(word, this.orig.word)) {
                this.freqBoost = this.orig.freqBoost;
                this.score = this.orig.score;
                return;
            }
            float dist = (float)this.orig.wordDist(word) / 2.0f;
            this.score = 1.0f - dist / (float)this.orig.length();
            if (this.metaphone.equals(this.orig.metaphone)) {
                this.score += 0.1f;
            }
            if (word.length() > 0 && this.orig.word.length() > 0 && word.charAt(0) == this.orig.word.charAt(0) && word.charAt(word.length() - 1) == this.orig.word.charAt(this.orig.word.length() - 1)) {
                this.score += 0.1f;
            }
            this.freqBoost = this.calcFreqBoost(SpellReader.this.freqSamples, freq);
            this.score += this.freqBoost;
        }

        public int length() {
            return this.word.length();
        }

        public boolean equals(Word other) {
            return this.word.equals(other.word);
        }

        public int wordDist(String other) {
            if (this.wordDist == null) {
                this.wordDist = new TRStringDistance2(this.word);
            }
            return this.wordDist.getDistance(other);
        }

        public int mphDist(String other) {
            if (this.mphDist == null) {
                this.mphDist = new TRStringDistance2(this.metaphone);
            }
            return this.mphDist.getDistance(other);
        }

        public String toString() {
            return this.word;
        }

        public void debug(PrintWriter w) {
            this.align(w, "word=" + this.word + "[" + this.orig.wordDist(this.word) + "]", 22);
            this.align(w, "mph=" + this.metaphone + "[" + this.orig.mphDist(this.metaphone) + "]", 13);
            this.align(w, "freq=" + this.freq, 12);
            if (this.orig != this && SpellReader.this.wordEquiv.isEquivalent(this.word, this.orig.word)) {
                this.align(w, "copyScore=" + this.orig.score, 20);
                w.println();
                return;
            }
            float dist = (float)this.orig.wordDist(this.word) / 2.0f;
            this.align(w, "base=" + (1.0f - dist / (float)this.orig.length()), 14);
            String mphStr = "0";
            if (this.metaphone.equals(this.orig.metaphone)) {
                mphStr = "0.1";
            }
            this.align(w, "mphBoost=" + mphStr, 13);
            String matchStr = "0";
            if (this.word.charAt(0) == this.orig.word.charAt(0) && this.word.charAt(this.word.length() - 1) == this.orig.word.charAt(this.orig.word.length() - 1)) {
                matchStr = "0.1";
            }
            this.align(w, "matchBoost=" + matchStr, 15);
            this.align(w, "freqBoost=" + this.freqBoost, 20);
            this.align(w, "totalScore=" + this.score, 22);
            w.println();
        }

        private void align(PrintWriter w, String s, int width) {
            w.print(s);
            for (int i = 0; i < width - s.length(); ++i) {
                w.print(" ");
            }
            w.print(" ");
        }

        private float calcFreqBoost(int[] termFreqs, int freq) {
            int i;
            if (freq == 0) {
                return -0.2f;
            }
            for (i = 0; i < 5 && freq < termFreqs[i]; ++i) {
            }
            if (i == 0) {
                return 0.25f;
            }
            int loFreq = i < 5 ? termFreqs[i] : 0;
            int hiFreq = termFreqs[i - 1];
            float loBoost = (float)(5 - i) * 0.05f;
            float boost = (float)((freq - loFreq) * 50 / (hiFreq - loFreq)) / 1000.0f + loBoost;
            return boost;
        }
    }
}

